<?php

namespace App\Filament\HospitalAdmin\Clusters\Pathology\Resources;

use Filament\Forms;
use Filament\Tables;
use Filament\Forms\Form;
use Filament\Tables\Table;
use App\Models\PathologyUnit;
use Filament\Resources\Resource;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Model;
use Filament\Pages\SubNavigationPosition;
use Illuminate\Database\Eloquent\Builder;
use App\Filament\HospitalAdmin\Clusters\Pathology;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\HospitalAdmin\Clusters\Pathology\Resources\PathologyUnitResource\Pages;
use App\Filament\HospitalAdmin\Clusters\Pathology\Resources\PathologyUnitResource\RelationManagers;
use App\Models\PathologyParameter;
use Filament\Notifications\Notification;

class PathologyUnitResource extends Resource
{
    protected static ?string $model = PathologyUnit::class;

    protected static ?string $cluster = Pathology::class;

    protected static SubNavigationPosition $subNavigationPosition = SubNavigationPosition::Top;

    protected static ?int $navigationSort = 2;

    public static function shouldRegisterNavigation(): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])  && !getModuleAccess('Pathology Categories') && !getModuleAccess('Pathology Tests')) {
            return false;
        } elseif (!auth()->user()->hasRole(['Admin', 'Receptionist']) && !getModuleAccess('Pathology Categories') && !getModuleAccess('Pathology Tests')) {
            return false;
        }
        return true;
    }

    public static function getNavigationLabel(): string
    {
        return __('messages.new_change.pathology_units');
    }

    public static function getLabel(): string
    {
        return __('messages.new_change.pathology_units');
    }

    public static function canCreate(): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])) {
            return true;
        }
        return false;
    }
    public static function canEdit(Model $record): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])) {
            return true;
        }
        return false;
    }

    public static function canDelete(Model $record): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])) {
            return true;
        }
        return false;
    }

    public static function canViewAny(): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])) {
            return true;
        }
        return false;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('name')
                    ->required()
                    ->validationAttribute(__('messages.common.name'))
                    ->maxLength(255),
            ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        $table = $table->modifyQueryUsing(function ($query) {
            return $query->where('tenant_id', getLoggedInUser()->tenant_id);
        });
        return $table
            ->defaultSort('id', 'desc')
            ->columns([
                TextColumn::make('name')
                    ->sortable()
                    ->searchable()
                    ->label(__('messages.common.name')),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->successNotificationTitle(__('messages.new_change.pathology_unit') . ' ' . __('messages.common.updated_successfully')),
                Tables\Actions\DeleteAction::make()
                    ->iconButton()
                    ->action(function (PathologyUnit $record) {
                        if (! canAccessRecord(PathologyUnit::class, $record->id)) {
                            return Notification::make()
                                ->danger()
                                ->title(__('messages.new_change.pathology_unit_not_found'))
                                ->send();
                        }

                        $pathologyParameterModels = [
                            PathologyParameter::class,
                        ];
                        $result = canDelete($pathologyParameterModels, 'unit_id', $record->id);
                        if ($result) {
                            return Notification::make()
                                ->danger()
                                ->title(__('messages.new_change.pathology_unit_cant_deleted'))
                                ->send();
                        }

                        $record->delete();
                        return Notification::make()
                            ->success()
                            ->title(__('messages.new_change.pathology_unit') . ' ' . __('messages.common.deleted_successfully'))
                            ->send();
                    }),
            ])
            ->recordAction(null)
            ->actionsColumnLabel(__('messages.common.action'))
            ->bulkActions([
                // Tables\Actions\BulkActionGroup::make([
                //     Tables\Actions\DeleteBulkAction::make(),
                // ]),
            ])
            ->emptyStateHeading(__('messages.common.no_data_found'));
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ManagePathologyUnits::route('/'),
        ];
    }
}
