<?php

namespace App\Filament\HospitalAdmin\Clusters\Pathology\Resources;

use Filament\Forms;
use Filament\Tables;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Filament\Resources\Resource;
use App\Models\PathologyCategory;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Model;
use Filament\Pages\SubNavigationPosition;
use Illuminate\Database\Eloquent\Builder;
use App\Filament\HospitalAdmin\Clusters\Pathology;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\HospitalAdmin\Clusters\Pathology\Resources\PathologyCategoryResource\Pages;
use App\Filament\HospitalAdmin\Clusters\Pathology\Resources\PathologyCategoryResource\RelationManagers;
use App\Models\PathologyTest;
use Filament\Notifications\Notification;

class PathologyCategoryResource extends Resource
{
    protected static ?string $model = PathologyCategory::class;

    protected static SubNavigationPosition $subNavigationPosition = SubNavigationPosition::Top;

    protected static ?int $navigationSort = 1;

    protected static ?string $cluster = Pathology::class;

    public static function shouldRegisterNavigation(): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])  && !getModuleAccess('Pathology Categories')) {
            return false;
        } elseif (!auth()->user()->hasRole(['Admin', 'Receptionist']) && !getModuleAccess('Pathology Categories')) {
            return false;
        }
        return true;
    }

    public static function getNavigationLabel(): string
    {
        return __('messages.pathology_categories');
    }

    public static function getLabel(): string
    {
        return __('messages.pathology_categories');
    }

    public static function canCreate(): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])) {
            return true;
        }
        return false;
    }

    public static function canEdit(Model $record): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])) {
            return true;
        }
        return false;
    }

    public static function canDelete(Model $record): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])) {
            return true;
        }
        return false;
    }

    public static function canViewAny(): bool
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist'])) {
            return true;
        }
        return false;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('name')
                    ->required()
                    ->validationAttribute(__('messages.common.name'))
                    ->maxLength(160)
            ])->columns(1);
    }


    public static function table(Table $table): Table
    {
        if (auth()->user()->hasRole(['Admin', 'Receptionist']) && !getModuleAccess('Pathology Categories')) {
            abort(404);
        }

        $table = $table->modifyQueryUsing(function ($query) {
            return $query->where('tenant_id', Auth::user()->tenant_id);
        });
        return $table
            ->recordUrl(null)
            ->recordAction(null)
            ->defaultSort('id', 'desc')
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->sortable()
                    ->label(__('messages.common.name'))
                    ->searchable(),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make()->iconButton()->modalWidth('md')->successNotificationTitle(__('messages.flash.pathology_category_updated')),
                Tables\Actions\DeleteAction::make()
                    ->iconButton()
                    ->action(function (PathologyCategory $record) {
                        if (! canAccessRecord(PathologyCategory::class, $record->id)) {
                            return Notification::make()
                                ->danger()
                                ->title(__('messages.flash.pathology_category_not_found'))
                                ->send();
                        }

                        $pathologyCategoryModels = [
                            PathologyTest::class,
                        ];
                        $result = canDelete($pathologyCategoryModels, 'category_id', $record->id);
                        if ($result) {
                            return Notification::make()
                                ->danger()
                                ->title(__('messages.flash.pathology_category_cant_deleted'))
                                ->send();
                        }

                        $record->delete();
                        return Notification::make()
                            ->success()
                            ->title(__('messages.flash.pathology_category_deleted'))
                            ->send();
                    }),
            ])->actionsColumnLabel(__('messages.common.action'))
            ->bulkActions([
                //
            ])
            ->emptyStateHeading(__('messages.common.no_data_found'));
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ManagePathologyCategories::route('/'),
        ];
    }
}
